/*
 * Vector.h
 *
 * Created 8/1/2009 By Johnny Huynh
 *
 * Version 00.00.01 8/1/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 #ifndef VECTOR_H
 #define VECTOR_H
 
 #include "math.h"
 #include "Math3D.h"
 #include "global.h"

 namespace Vector
 {
    template <typename T> inline T get_angle_degrees( T x, T y );
    template <typename T> inline T get_angle_degrees_standard( T x, T y );
    template <typename T> inline VECTOR2_TYPE get_xy_direction( const T& heading_degrees );
    template <typename T> inline VECTOR2_TYPE get_xy_direction_normalized( const T& heading_degrees );
    template <typename T> inline VECTOR2_TYPE get_xy_direction_standard( const T& degrees );
    template <typename T> inline VECTOR2_TYPE get_xy_direction_standard_normalized( const T& degrees );
    template <typename T> inline T get_magnitude( const VECTOR3_TYPE& v );
    template <typename T> inline T get_magnitude_squared( const VECTOR3_TYPE& v );
    template <typename T> inline VECTOR3_TYPE get_normal( const VECTOR3_TYPE& v );
    template <typename T> inline void normalize( T& x, T& y );
    inline VECTOR2_TYPE& normalize( VECTOR2_TYPE& v );
 }

 /**
  * get_angle_degrees() returns the angle in degrees for the
  * specified 2-dimensional vector. This function assumes the
  * geographical coordinate system. The angle returned is
  * between 0 (inclusive) and 360 (exclusive) degrees. The
  * specified vector does not have to be normalized.
  *
  * @param (T) x
  * @param (T) y
  * @return T
  */
 template <typename T>
 inline T Vector::get_angle_degrees( T x, T y )
 {
    Vector::normalize( x, y );
    
    // Geographic coordinate system (-x, z, y)
    // y is the direction facing
    // z is the vertical direction
    //        y, x
    // quad1: +, +
    // quad2: -, +
    // quad3: -, -
    // quad4: +, -
    T radians( acos( y ) );
    if ( x > ZERO ) // if x is positive
    {
        radians = (2.0f*PI) - radians;//-radians; //(2.0f*PI) - radians;
    }
    return Math3D::radians_to_degrees( radians );
    
    //return Vector::get_angle_degrees_standard( y, -x );
 }
 
 /**
  * get_angle_degrees_standard() returns the angle in degrees for the
  * specified 2-dimensional vector. This function assumes the
  * standard XYZ-coordinate system. The angle returned is
  * between 0 (inclusive) and 360 (exclusive) degrees.
  *
  * @param (T) x
  * @param (T) y
  * @return T
  */
 template <typename T>
 inline T Vector::get_angle_degrees_standard( T x, T y )
 {
    Vector::normalize( x, y );
    
    // Standard coordinate system (x, y, z)
    // x is the direction facing
    // y is the vertical direction
    //        x, y
    // quad1: +, +
    // quad2: -, +
    // quad3: -, -
    // quad4: +, -
    T radians( acos( x ) );
    if ( y < ZERO ) // if y is negative
    {
        radians = (2.0f*PI) - radians;//-radians; //(2.0f*PI) - radians;
    }
    return Math3D::radians_to_degrees( radians );
 }
 
 /**
  * get_xy_direction() returns a vector representing the xy-direction of
  * the specified heading angle in degrees. The geographical coordinate 
  * system is assumed. The vector returned is NOT normalized.
  *
  * @param (const T&) heading_degrees
  * @return VECTOR2_TYPE
  */
 template <typename T> 
 inline VECTOR2_TYPE Vector::get_xy_direction( const T& heading_degrees )
 {
    //return Vector::get_xy_direction_standard( heading_degrees + 90.0f );
    
    // geographical coordinate system location (-x, y) at height z
    // gc_x = -x, gc_y = z, gc_z = y  (i.e. (-x, z, y) )
    // the -x axis is the direction left of an object
    // the +y axis is the direction an object should be facing
    // the +z axis is the up direction
    T radians( Math3D::degrees_to_radians( heading_degrees ) );
    return VECTOR2_TYPE( -sin( radians ), cos( radians ) ); // negative sine because using geographical coordinate system (-x, z, y)
 }
 
 /**
  * get_xy_direction_normalized() returns a vector representing the xy-direction of
  * the specified heading angle in degrees. The geographical coordinate 
  * system is assumed. The vector returned is normalized.
  *
  * @param (const T&) heading_degrees
  * @return VECTOR2_TYPE
  */
 template <typename T> 
 inline VECTOR2_TYPE Vector::get_xy_direction_normalized( const T& heading_degrees )
 {
    //VECTOR2_TYPE v( Vector::get_xy_direction( heading_degrees ) );
    //printf( "%1f\n", sqrt( (v.get_x()*v.get_x()) + (v.get_y()*v.get_y()) ) );
    return Vector::normalize( Vector::get_xy_direction( heading_degrees ) );
 }
 
 /**
  * get_xy_direction_standard() returns a vector representing the xy-direction of
  * the specified angle in degrees. The standard xyz-coordinate system is
  * assumed. The vector returned is NOT normalized.
  *
  * @param (const T&) heading_degrees
  * @return VECTOR2_TYPE
  */
 template <typename T> 
 inline VECTOR2_TYPE Vector::get_xy_direction_standard( const T& degrees )
 {
    T radians( Math3D::degrees_to_radians( degrees ) );
    return VECTOR2_TYPE( cos( radians ), sin( radians ) );
 }
 
 /**
  * get_xy_direction_standard_normalized() returns a vector representing 
  * the xy-direction of the specified heading angle in degrees. The standard 
  * xyz-coordinate system is assumed. The vector returned is normalized.
  *
  * @param (const T&) heading_degrees
  * @return VECTOR2_TYPE
  */
 template <typename T> 
 inline VECTOR2_TYPE Vector::get_xy_direction_standard_normalized( const T& degrees )
 {
    return Vector::normalize<T>( Vector::get_xy_direction_standard( degrees ) );
 }
 
 /**
  * get_magnitude() returns the magnitude of the specified vector.
  *
  * @param (const VECTOR3_TYPE&) v
  * @return T
  */
 template <typename T> 
 inline T Vector::get_magnitude( const VECTOR3_TYPE& v )
 {
    return sqrt( (v.get_x()*v.get_x()) + (v.get_y()*v.get_y()) + (v.get_z()*v.get_z()) );
 }
 
 /**
  * get_magnitude_squared() returns the magnitude squared of the specified vector.
  *
  * @param (const VECTOR3_TYPE&) v
  * @return T
  */
 template <typename T> 
 inline T Vector::get_magnitude_squared( const VECTOR3_TYPE& v )
 {
    return (v.get_x()*v.get_x()) + (v.get_y()*v.get_y()) + (v.get_z()*v.get_z());
 }
 
 /**
  * get_normal() returns the normalized vector of the specified vector.
  *
  * @param (const VECTOR3_TYPE&) v
  * @return VECTOR3_TYPE
  */
 template <typename T>
 inline VECTOR3_TYPE Vector::get_normal( const VECTOR3_TYPE& v )
 {
    T magnitude_squared( Vector::get_magnitude_squared<T>( v ) );
    
    // if v is not normalized yet
    if ( magnitude_squared != ONE )
        return v / sqrt( magnitude_squared );
    else // v is already normalized
        return v;
 }
 
 /**
  * normalize() normalizes the specified 2-dimensional vector.
  *
  * @param (T&) x
  * @param (T&) y
  */
 template <typename T>
 inline void Vector::normalize( T& x, T& y )
 {
    T m( (x*x) + (y*y) );
    
    // prevent divide by zero
    if ( m == ZERO )
        return;
    
    if ( m != ONE )
    {
        m = sqrt( m );
        x /= m;
        y /= m;
    }
 }
 
 /**
  * normalize() normalizes the specified 2-dimensional vector,
  * and returns the normalized vector.
  *
  * @param (VECTOR2_TYPE&) v
  * @return VECTOR2_TYPE&
  */
 inline VECTOR2_TYPE& Vector::normalize( VECTOR2_TYPE& v )
 {
    v.normalize();
    return v;
 }

 #endif // VECTOR_H